<?php
/**
 * E-Kitap Veritabanı Migration
 * Tüm tabloları oluşturur + seed data ekler
 *
 * Kullanım: php migrations/001_initial.php
 */

require_once __DIR__ . '/../core/Database.php';

// Config'i yükle
$dbConfig = require __DIR__ . '/../config/database.php';

try {
    $dsn = "mysql:host={$dbConfig['host']};charset={$dbConfig['charset']}";
    $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password'], $dbConfig['options']);

    // Veritabanını oluştur
    $dbName = $dbConfig['dbname'];
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    $pdo->exec("USE `{$dbName}`");

    echo "✓ Veritabanı '{$dbName}' hazır.\n\n";

    // ──────────────────────────────────────
    // USERS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `users` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `email` VARCHAR(255) NOT NULL,
            `password_hash` VARCHAR(255) NOT NULL,
            `name` VARCHAR(150) NOT NULL DEFAULT '',
            `locale` VARCHAR(5) NOT NULL DEFAULT 'tr',
            `avatar_url` VARCHAR(500) NULL,
            `is_active` TINYINT(1) NOT NULL DEFAULT 1,
            `last_login_at` DATETIME NULL,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY `uk_users_email` (`email`),
            KEY `idx_users_active` (`is_active`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ users tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // ROLES
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `roles` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `name` VARCHAR(50) NOT NULL,
            `description` VARCHAR(255) NULL,
            UNIQUE KEY `uk_roles_name` (`name`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ roles tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // USER_ROLES
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `user_roles` (
            `user_id` INT UNSIGNED NOT NULL,
            `role_id` INT UNSIGNED NOT NULL,
            PRIMARY KEY (`user_id`, `role_id`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
            FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ user_roles tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // CATEGORIES
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `categories` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `slug` VARCHAR(100) NOT NULL,
            `icon` VARCHAR(50) NULL,
            `sort_order` INT NOT NULL DEFAULT 0,
            `is_active` TINYINT(1) NOT NULL DEFAULT 1,
            UNIQUE KEY `uk_categories_slug` (`slug`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ categories tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // CATEGORY_TRANSLATIONS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `category_translations` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `category_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL,
            `name` VARCHAR(200) NOT NULL,
            UNIQUE KEY `uk_cat_trans` (`category_id`, `lang`),
            FOREIGN KEY (`category_id`) REFERENCES `categories`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ category_translations tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // BOOKS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `books` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `slug` VARCHAR(200) NOT NULL,
            `category_id` INT UNSIGNED NULL,
            `status` ENUM('draft','published','hidden') NOT NULL DEFAULT 'draft',
            `is_free` TINYINT(1) NOT NULL DEFAULT 0,
            `preview_pages` INT NOT NULL DEFAULT 10,
            `default_lang` VARCHAR(5) NOT NULL DEFAULT 'tr',
            `cover_path` VARCHAR(500) NULL,
            `total_pages` INT NOT NULL DEFAULT 0,
            `sort_order` INT NOT NULL DEFAULT 0,
            `view_count` INT UNSIGNED NOT NULL DEFAULT 0,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY `uk_books_slug` (`slug`),
            KEY `idx_books_status` (`status`),
            KEY `idx_books_category` (`category_id`),
            KEY `idx_books_free` (`is_free`),
            FOREIGN KEY (`category_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ books tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // BOOK_TRANSLATIONS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `book_translations` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL,
            `title` VARCHAR(500) NOT NULL,
            `description` TEXT NULL,
            `author` VARCHAR(300) NULL,
            `publisher` VARCHAR(300) NULL,
            `isbn` VARCHAR(20) NULL,
            `seo_title` VARCHAR(200) NULL,
            `seo_desc` VARCHAR(500) NULL,
            `tags` VARCHAR(500) NULL COMMENT 'Virgülle ayrılmış etiketler',
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY `uk_book_trans` (`book_id`, `lang`),
            KEY `idx_book_trans_lang` (`lang`),
            FULLTEXT KEY `ft_book_search` (`title`, `description`, `author`, `tags`),
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ book_translations tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // BOOK_ASSETS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `book_assets` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL,
            `format` ENUM('chunk','epub','pdf') NOT NULL,
            `version` INT NOT NULL DEFAULT 1,
            `path` VARCHAR(500) NOT NULL COMMENT 'Dosya veya base dizin yolu',
            `preview_path` VARCHAR(500) NULL COMMENT 'EPUB/PDF preview dosyası',
            `is_active` TINYINT(1) NOT NULL DEFAULT 1,
            `size_bytes` BIGINT UNSIGNED NULL,
            `checksum` VARCHAR(64) NULL COMMENT 'SHA-256 hash',
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY `idx_book_assets` (`book_id`, `lang`, `format`, `is_active`),
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ book_assets tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // BOOK_CHUNK_INDEX
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `book_chunk_index` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL,
            `version` INT NOT NULL DEFAULT 1,
            `chunk_no` INT NOT NULL,
            `page_start` INT NOT NULL,
            `page_end` INT NOT NULL,
            `file_path` VARCHAR(500) NOT NULL,
            UNIQUE KEY `uk_chunk` (`book_id`, `lang`, `version`, `chunk_no`),
            KEY `idx_chunk_pages` (`book_id`, `lang`, `version`, `page_start`, `page_end`),
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ book_chunk_index tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // SUBSCRIPTIONS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `subscriptions` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NOT NULL,
            `provider` VARCHAR(30) NOT NULL DEFAULT 'google_play',
            `product_id` VARCHAR(100) NOT NULL,
            `order_id` VARCHAR(200) NULL,
            `purchase_token_hash` VARCHAR(64) NOT NULL COMMENT 'SHA-256 hash',
            `is_active` TINYINT(1) NOT NULL DEFAULT 0,
            `starts_at` DATETIME NULL,
            `expiry_time` DATETIME NULL,
            `auto_renewing` TINYINT(1) NOT NULL DEFAULT 0,
            `cancel_reason` VARCHAR(50) NULL,
            `last_verified_at` DATETIME NULL,
            `raw_response` JSON NULL COMMENT 'Son doğrulama yanıtı',
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY `idx_sub_user` (`user_id`, `is_active`),
            KEY `idx_sub_expiry` (`expiry_time`),
            KEY `idx_sub_token` (`purchase_token_hash`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ subscriptions tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // READING_PROGRESS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `reading_progress` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NOT NULL,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL DEFAULT 'tr',
            `last_format` ENUM('chunk','epub','pdf') NOT NULL DEFAULT 'chunk',
            `last_page` INT NOT NULL DEFAULT 0,
            `last_location` TEXT NULL COMMENT 'EPUB CFI veya PDF offset',
            `progress_percent` DECIMAL(5,2) NOT NULL DEFAULT 0.00,
            `total_reading_time` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'Saniye cinsinden',
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY `uk_progress` (`user_id`, `book_id`, `lang`),
            KEY `idx_progress_user` (`user_id`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ reading_progress tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // BOOKMARKS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `bookmarks` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NOT NULL,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL DEFAULT 'tr',
            `page` INT NULL,
            `location` TEXT NULL,
            `label` VARCHAR(300) NULL,
            `color` VARCHAR(20) NULL DEFAULT '#FFD700',
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY `idx_bm_user_book` (`user_id`, `book_id`, `lang`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ bookmarks tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // NOTES (Altı çizme / notlar)
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `notes` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NOT NULL,
            `book_id` INT UNSIGNED NOT NULL,
            `lang` VARCHAR(5) NOT NULL DEFAULT 'tr',
            `page` INT NULL,
            `location` TEXT NULL,
            `selected_text` TEXT NULL,
            `note_content` TEXT NULL,
            `highlight_color` VARCHAR(20) NULL DEFAULT '#FFFF00',
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY `idx_notes_user_book` (`user_id`, `book_id`, `lang`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
            FOREIGN KEY (`book_id`) REFERENCES `books`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ notes tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // REFRESH_TOKENS
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `refresh_tokens` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NOT NULL,
            `token_hash` VARCHAR(64) NOT NULL,
            `expires_at` DATETIME NOT NULL,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY `uk_refresh_token` (`token_hash`),
            KEY `idx_refresh_user` (`user_id`),
            KEY `idx_refresh_expires` (`expires_at`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ refresh_tokens tablosu oluşturuldu.\n";

    // ──────────────────────────────────────
    // SUBSCRIPTION_LOGS (Audit)
    // ──────────────────────────────────────
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `subscription_logs` (
            `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT UNSIGNED NULL,
            `action` VARCHAR(50) NOT NULL,
            `product_id` VARCHAR(100) NULL,
            `status` VARCHAR(30) NULL,
            `details` JSON NULL,
            `ip_address` VARCHAR(45) NULL,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY `idx_sub_log_user` (`user_id`),
            KEY `idx_sub_log_date` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ subscription_logs tablosu oluşturuldu.\n";

    // ══════════════════════════════════════
    // SEED DATA
    // ══════════════════════════════════════
    echo "\n── Seed Data ──\n";

    // Roller
    $roles = ['superAdmin', 'admin', 'editor', 'translator', 'support'];
    $descriptions = [
        'Tam yetki - sistem yöneticisi',
        'İçerik ve kullanıcı yönetimi',
        'İçerik düzenleme',
        'Çeviri yönetimi',
        'Destek ekibi'
    ];
    foreach ($roles as $i => $role) {
        $stmt = $pdo->prepare("INSERT IGNORE INTO roles (name, description) VALUES (?, ?)");
        $stmt->execute([$role, $descriptions[$i]]);
    }
    echo "✓ Roller eklendi.\n";

    // Varsayılan admin kullanıcı
    $adminEmail = 'admin@ekitap.com';
    $adminPass = password_hash('Admin123!', PASSWORD_BCRYPT);
    $stmt = $pdo->prepare("INSERT IGNORE INTO users (email, password_hash, name, locale) VALUES (?, ?, ?, ?)");
    $stmt->execute([$adminEmail, $adminPass, 'Admin', 'tr']);

    $adminUser = $pdo->query("SELECT id FROM users WHERE email = '{$adminEmail}'")->fetch();
    if ($adminUser) {
        $adminRoleId = $pdo->query("SELECT id FROM roles WHERE name = 'superAdmin'")->fetch()['id'];
        $pdo->prepare("INSERT IGNORE INTO user_roles (user_id, role_id) VALUES (?, ?)")
            ->execute([$adminUser['id'], $adminRoleId]);
    }
    echo "✓ Admin kullanıcı oluşturuldu (admin@ekitap.com / Admin123!).\n";

    // Örnek kategoriler
    $categories = [
        ['slug' => 'roman', 'icon' => 'book'],
        ['slug' => 'bilim', 'icon' => 'science'],
        ['slug' => 'tarih', 'icon' => 'history'],
        ['slug' => 'felsefe', 'icon' => 'psychology'],
        ['slug' => 'kisisel-gelisim', 'icon' => 'trending_up'],
        ['slug' => 'teknoloji', 'icon' => 'computer'],
        ['slug' => 'cocuk', 'icon' => 'child_care'],
        ['slug' => 'edebiyat', 'icon' => 'menu_book'],
    ];

    $catTranslations = [
        'roman' => ['tr' => 'Roman', 'en' => 'Novel', 'de' => 'Roman', 'es' => 'Novela', 'fr' => 'Roman', 'it' => 'Romanzo', 'pt' => 'Romance', 'ar' => 'رواية'],
        'bilim' => ['tr' => 'Bilim', 'en' => 'Science', 'de' => 'Wissenschaft', 'es' => 'Ciencia', 'fr' => 'Science', 'it' => 'Scienza', 'pt' => 'Ciência', 'ar' => 'علم'],
        'tarih' => ['tr' => 'Tarih', 'en' => 'History', 'de' => 'Geschichte', 'es' => 'Historia', 'fr' => 'Histoire', 'it' => 'Storia', 'pt' => 'História', 'ar' => 'تاريخ'],
        'felsefe' => ['tr' => 'Felsefe', 'en' => 'Philosophy', 'de' => 'Philosophie', 'es' => 'Filosofía', 'fr' => 'Philosophie', 'it' => 'Filosofia', 'pt' => 'Filosofia', 'ar' => 'فلسفة'],
        'kisisel-gelisim' => ['tr' => 'Kişisel Gelişim', 'en' => 'Self-Help', 'de' => 'Persönliche Entwicklung', 'es' => 'Autoayuda', 'fr' => 'Développement personnel', 'it' => 'Crescita personale', 'pt' => 'Autoajuda', 'ar' => 'تطوير الذات'],
        'teknoloji' => ['tr' => 'Teknoloji', 'en' => 'Technology', 'de' => 'Technologie', 'es' => 'Tecnología', 'fr' => 'Technologie', 'it' => 'Tecnologia', 'pt' => 'Tecnologia', 'ar' => 'تكنولوجيا'],
        'cocuk' => ['tr' => 'Çocuk', 'en' => 'Children', 'de' => 'Kinder', 'es' => 'Niños', 'fr' => 'Enfants', 'it' => 'Bambini', 'pt' => 'Crianças', 'ar' => 'أطفال'],
        'edebiyat' => ['tr' => 'Edebiyat', 'en' => 'Literature', 'de' => 'Literatur', 'es' => 'Literatura', 'fr' => 'Littérature', 'it' => 'Letteratura', 'pt' => 'Literatura', 'ar' => 'أدب'],
    ];

    foreach ($categories as $i => $cat) {
        $stmt = $pdo->prepare("INSERT IGNORE INTO categories (slug, icon, sort_order) VALUES (?, ?, ?)");
        $stmt->execute([$cat['slug'], $cat['icon'], $i]);

        $catRow = $pdo->query("SELECT id FROM categories WHERE slug = '{$cat['slug']}'")->fetch();
        if ($catRow && isset($catTranslations[$cat['slug']])) {
            foreach ($catTranslations[$cat['slug']] as $lang => $name) {
                $pdo->prepare("INSERT IGNORE INTO category_translations (category_id, lang, name) VALUES (?, ?, ?)")
                    ->execute([$catRow['id'], $lang, $name]);
            }
        }
    }
    echo "✓ Kategoriler ve çevirileri (8 dil) eklendi.\n";

    echo "\n══════════════════════════════\n";
    echo "✅ Migration tamamlandı!\n";
    echo "══════════════════════════════\n";

} catch (PDOException $e) {
    echo "❌ Hata: " . $e->getMessage() . "\n";
    exit(1);
}
