<?php
/**
 * Basit Router - cPanel uyumlu
 * .htaccess ile tüm istekler index.php'ye yönlendirilir
 */
class Router
{
    private array $routes = [];
    private array $middlewares = [];
    private string $prefix = '';

    /**
     * Route grubu (prefix + middleware)
     */
    public function group(string $prefix, array $middlewares, callable $callback): void
    {
        $previousPrefix = $this->prefix;
        $previousMiddlewares = $this->middlewares;

        $this->prefix .= $prefix;
        $this->middlewares = array_merge($this->middlewares, $middlewares);

        $callback($this);

        $this->prefix = $previousPrefix;
        $this->middlewares = $previousMiddlewares;
    }

    public function get(string $path, string $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }

    public function post(string $path, string $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }

    public function put(string $path, string $handler): void
    {
        $this->addRoute('PUT', $path, $handler);
    }

    public function delete(string $path, string $handler): void
    {
        $this->addRoute('DELETE', $path, $handler);
    }

    private function addRoute(string $method, string $path, string $handler): void
    {
        $fullPath = $this->prefix . $path;
        $this->routes[] = [
            'method'      => $method,
            'path'        => $fullPath,
            'handler'     => $handler,
            'middlewares'  => $this->middlewares,
        ];
    }

    /**
     * İsteği eşleştir ve çalıştır
     */
    public function dispatch(string $method, string $uri): void
    {
        // Query string'i kaldır
        $uri = strtok($uri, '?');
        $uri = rtrim($uri, '/');
        if ($uri === '') $uri = '/';

        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) continue;

            $params = $this->matchRoute($route['path'], $uri);
            if ($params !== false) {
                // Middleware'leri çalıştır
                foreach ($route['middlewares'] as $middleware) {
                    $mw = new $middleware();
                    $result = $mw->handle();
                    if ($result === false) return;
                }

                // Controller@method çağır
                [$controllerClass, $methodName] = explode('@', $route['handler']);
                $controller = new $controllerClass();
                $controller->$methodName($params);
                return;
            }
        }

        // 404
        Response::json(['error' => 'Endpoint bulunamadı', 'code' => 'NOT_FOUND'], 404);
    }

    /**
     * Route pattern eşleştirme ({id}, {slug} vb.)
     */
    private function matchRoute(string $pattern, string $uri): array|false
    {
        $patternParts = explode('/', trim($pattern, '/'));
        $uriParts = explode('/', trim($uri, '/'));

        if (count($patternParts) !== count($uriParts)) return false;

        $params = [];
        for ($i = 0; $i < count($patternParts); $i++) {
            if (preg_match('/^\{(\w+)\}$/', $patternParts[$i], $m)) {
                $params[$m[1]] = urldecode($uriParts[$i]);
            } elseif ($patternParts[$i] !== $uriParts[$i]) {
                return false;
            }
        }

        return $params;
    }
}
