<?php
/**
 * Request Helper - Gelen istek verilerini parse eder
 */
class Request
{
    private static ?array $jsonBody = null;

    /**
     * JSON body'yi al
     */
    public static function body(): array
    {
        if (self::$jsonBody === null) {
            $raw = file_get_contents('php://input');
            self::$jsonBody = json_decode($raw, true) ?? [];
        }
        return self::$jsonBody;
    }

    /**
     * Body'den belirli bir alan
     */
    public static function input(string $key, mixed $default = null): mixed
    {
        return self::body()[$key] ?? $default;
    }

    /**
     * Query parametresi
     */
    public static function query(string $key, mixed $default = null): mixed
    {
        return $_GET[$key] ?? $default;
    }

    /**
     * Query'den int al
     */
    public static function queryInt(string $key, int $default = 0): int
    {
        return (int)($_GET[$key] ?? $default);
    }

    /**
     * Authorization header'dan Bearer token al
     */
    public static function bearerToken(): ?string
    {
        $header = $_SERVER['HTTP_AUTHORIZATION']
            ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION']
            ?? '';

        if (preg_match('/Bearer\s+(.+)/i', $header, $m)) {
            return trim($m[1]);
        }
        return null;
    }

    /**
     * HTTP method
     */
    public static function method(): string
    {
        return strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
    }

    /**
     * Request URI (path only)
     */
    public static function uri(): string
    {
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        return strtok($uri, '?');
    }

    /**
     * Client IP adresi
     */
    public static function ip(): string
    {
        return $_SERVER['HTTP_X_FORWARDED_FOR']
            ?? $_SERVER['HTTP_X_REAL_IP']
            ?? $_SERVER['REMOTE_ADDR']
            ?? '0.0.0.0';
    }

    /**
     * Dosya yükleme (multipart form)
     */
    public static function file(string $key): ?array
    {
        return $_FILES[$key] ?? null;
    }

    /**
     * Zorunlu alanları kontrol et
     */
    public static function validate(array $required): array
    {
        $body = self::body();
        $errors = [];
        foreach ($required as $field) {
            if (!isset($body[$field]) || (is_string($body[$field]) && trim($body[$field]) === '')) {
                $errors[] = "$field alanı zorunludur";
            }
        }
        if (!empty($errors)) {
            Response::error('Validasyon hatası', 'VALIDATION_ERROR', 422, $errors);
        }
        return $body;
    }
}
