<?php
/**
 * Subscription Controller - Abonelik doğrulama ve entitlement
 */
class SubscriptionController
{
    /**
     * POST /api/subscription/verify
     * Google Play satın alma doğrulaması
     */
    public function verify(array $params): void
    {
        global $currentUser;

        $data = Request::validate(['productId', 'purchaseToken', 'packageName']);

        $productId = $data['productId'];
        $purchaseToken = $data['purchaseToken'];
        $packageName = $data['packageName'];

        $config = require __DIR__ . '/../config/app.php';

        // Package name kontrolü
        if ($packageName !== $config['google_play']['package_name']) {
            $this->logAction($currentUser['id'], 'verify_failed', $productId, 'invalid_package');
            Response::error('Geçersiz paket adı', 'INVALID_PACKAGE', 400);
        }

        // Google Play Developer API ile doğrulama
        $verification = $this->verifyWithGoogle($packageName, $productId, $purchaseToken, $config);

        if (!$verification['success']) {
            $this->logAction($currentUser['id'], 'verify_failed', $productId, 'google_api_error', $verification);
            Response::error(
                'Abonelik doğrulanamadı: ' . ($verification['error'] ?? 'Bilinmeyen hata'),
                'VERIFICATION_FAILED',
                400
            );
        }

        $subscriptionData = $verification['data'];
        $tokenHash = hash('sha256', $purchaseToken);

        // Mevcut aboneliği kontrol et
        $existing = Database::fetch(
            'SELECT id FROM subscriptions WHERE purchase_token_hash = ?',
            [$tokenHash]
        );

        $expiryTime = null;
        if (isset($subscriptionData['expiryTimeMillis'])) {
            $expiryTime = date('Y-m-d H:i:s', $subscriptionData['expiryTimeMillis'] / 1000);
        }

        $isActive = ($subscriptionData['paymentState'] ?? 0) == 1
            && (($subscriptionData['expiryTimeMillis'] ?? 0) / 1000) > time();

        $autoRenewing = $subscriptionData['autoRenewing'] ?? false;
        $orderId = $subscriptionData['orderId'] ?? null;

        if ($existing) {
            // Güncelle
            Database::execute(
                'UPDATE subscriptions SET
                    is_active = ?, expiry_time = ?, auto_renewing = ?,
                    order_id = ?, last_verified_at = NOW(), raw_response = ?,
                    updated_at = NOW()
                 WHERE id = ?',
                [
                    $isActive ? 1 : 0, $expiryTime, $autoRenewing ? 1 : 0,
                    $orderId, json_encode($subscriptionData), $existing['id']
                ]
            );
        } else {
            // Yeni kayıt
            Database::insert(
                'INSERT INTO subscriptions
                    (user_id, provider, product_id, order_id, purchase_token_hash,
                     is_active, starts_at, expiry_time, auto_renewing, last_verified_at, raw_response)
                 VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, NOW(), ?)',
                [
                    $currentUser['id'], 'google_play', $productId, $orderId,
                    $tokenHash, $isActive ? 1 : 0, $expiryTime,
                    $autoRenewing ? 1 : 0, json_encode($subscriptionData)
                ]
            );
        }

        $this->logAction($currentUser['id'], 'verify_success', $productId, $isActive ? 'active' : 'inactive');

        Response::success([
            'isActive'     => $isActive,
            'productId'    => $productId,
            'expiryTime'   => $expiryTime,
            'autoRenewing' => $autoRenewing,
        ], 'Abonelik doğrulandı');
    }

    /**
     * GET /api/me/entitlement
     * Kullanıcının güncel erişim hakları
     */
    public function entitlement(array $params): void
    {
        global $currentUser;

        $subscription = Database::fetch(
            'SELECT product_id, is_active, expiry_time, auto_renewing, last_verified_at
             FROM subscriptions
             WHERE user_id = ? AND is_active = 1 AND expiry_time > NOW()
             ORDER BY expiry_time DESC LIMIT 1',
            [$currentUser['id']]
        );

        $hasActiveSubscription = $subscription !== null;

        Response::success([
            'userId'              => $currentUser['id'],
            'hasActiveSubscription' => $hasActiveSubscription,
            'subscription'        => $subscription ? [
                'productId'      => $subscription['product_id'],
                'expiryTime'     => $subscription['expiry_time'],
                'autoRenewing'   => (bool)$subscription['auto_renewing'],
                'lastVerifiedAt' => $subscription['last_verified_at'],
            ] : null,
            'accessLevel' => $hasActiveSubscription ? 'premium' : 'free',
        ]);
    }

    /**
     * Google Play Developer API ile doğrulama
     */
    private function verifyWithGoogle(string $packageName, string $productId, string $purchaseToken, array $config): array
    {
        $serviceAccountFile = $config['google_play']['service_account_json'];

        // Service account dosyası yoksa mock yanıt (development)
        if (!file_exists($serviceAccountFile)) {
            // Development modunda simüle et
            if ($config['debug'] ?? false) {
                return [
                    'success' => true,
                    'data' => [
                        'orderId'          => 'GPA.DEV-' . time(),
                        'startTimeMillis'  => time() * 1000,
                        'expiryTimeMillis' => (time() + 30 * 86400) * 1000,
                        'autoRenewing'     => true,
                        'paymentState'     => 1,
                        'developerPayload' => '',
                    ],
                ];
            }

            return [
                'success' => false,
                'error'   => 'Google service account yapılandırması eksik',
            ];
        }

        try {
            // Google OAuth2 token al
            $accessToken = $this->getGoogleAccessToken($serviceAccountFile);

            if (!$accessToken) {
                return ['success' => false, 'error' => 'Google API erişim tokeni alınamadı'];
            }

            // Subscription bilgisi al
            $url = "https://androidpublisher.googleapis.com/androidpublisher/v3/applications/{$packageName}/purchases/subscriptions/{$productId}/tokens/{$purchaseToken}";

            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER     => ["Authorization: Bearer {$accessToken}"],
                CURLOPT_TIMEOUT        => 30,
                CURLOPT_SSL_VERIFYPEER => true,
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200) {
                return [
                    'success' => false,
                    'error'   => "Google API yanıtı: HTTP {$httpCode}",
                    'raw'     => $response,
                ];
            }

            $data = json_decode($response, true);
            return ['success' => true, 'data' => $data];

        } catch (\Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Google Service Account ile OAuth2 token al
     */
    private function getGoogleAccessToken(string $serviceAccountFile): ?string
    {
        $sa = json_decode(file_get_contents($serviceAccountFile), true);

        if (!$sa || !isset($sa['client_email'], $sa['private_key'])) {
            return null;
        }

        $now = time();
        $header = base64_encode(json_encode(['alg' => 'RS256', 'typ' => 'JWT']));
        $claim = base64_encode(json_encode([
            'iss'   => $sa['client_email'],
            'scope' => 'https://www.googleapis.com/auth/androidpublisher',
            'aud'   => 'https://oauth2.googleapis.com/token',
            'iat'   => $now,
            'exp'   => $now + 3600,
        ]));

        $signatureInput = "{$header}.{$claim}";
        $privateKey = openssl_pkey_get_private($sa['private_key']);
        openssl_sign($signatureInput, $signature, $privateKey, OPENSSL_ALGO_SHA256);

        $jwt = $signatureInput . '.' . base64_encode($signature);

        // Token isteği
        $ch = curl_init('https://oauth2.googleapis.com/token');
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query([
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion'  => $jwt,
            ]),
        ]);

        $response = json_decode(curl_exec($ch), true);
        curl_close($ch);

        return $response['access_token'] ?? null;
    }

    /**
     * Abonelik log kaydı
     */
    private function logAction(int $userId, string $action, ?string $productId, string $status, ?array $details = null): void
    {
        Database::insert(
            'INSERT INTO subscription_logs (user_id, action, product_id, status, details, ip_address)
             VALUES (?, ?, ?, ?, ?, ?)',
            [$userId, $action, $productId, $status, $details ? json_encode($details) : null, Request::ip()]
        );
    }
}
