<?php
/**
 * Admin Book Controller - Kitap CRUD ve çeviri yönetimi
 */
class AdminBookController
{
    /**
     * GET /api/admin/books
     */
    public function index(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';
        $page  = max(1, Request::queryInt('page', 1));
        $limit = min(100, max(1, Request::queryInt('limit', 20)));
        $offset = ($page - 1) * $limit;
        $status = Request::query('status', '');
        $lang = Request::query('lang', $config['default_language']);

        $where = ['1=1'];
        $bindings = [];

        if ($status) {
            $where[] = 'b.status = ?';
            $bindings[] = $status;
        }

        $whereClause = implode(' AND ', $where);
        $total = (int)Database::fetch(
            "SELECT COUNT(*) as c FROM books b WHERE {$whereClause}", $bindings
        )['c'];

        $books = Database::fetchAll(
            "SELECT b.*, bt.title, bt.author, c.slug as category_slug
             FROM books b
             LEFT JOIN book_translations bt ON bt.book_id = b.id AND bt.lang = ?
             LEFT JOIN categories c ON c.id = b.category_id
             WHERE {$whereClause}
             ORDER BY b.updated_at DESC
             LIMIT ? OFFSET ?",
            array_merge([$lang], $bindings, [$limit, $offset])
        );

        // Her kitap için format bilgisi
        foreach ($books as &$book) {
            $assets = Database::fetchAll(
                'SELECT DISTINCT format FROM book_assets WHERE book_id = ? AND is_active = 1',
                [$book['id']]
            );
            $book['formats'] = array_column($assets, 'format');

            $langCount = Database::fetch(
                'SELECT COUNT(*) as c FROM book_translations WHERE book_id = ?',
                [$book['id']]
            )['c'];
            $book['translationCount'] = (int)$langCount;
        }

        Response::paginated($books, $total, $page, $limit);
    }

    /**
     * POST /api/admin/books
     */
    public function store(array $params): void
    {
        $data = Request::validate(['slug', 'title']);

        $slug = trim($data['slug']);
        $title = trim($data['title']);
        $lang = Request::input('lang', 'tr');

        // Slug tekrarı kontrolü
        if (Database::fetch('SELECT id FROM books WHERE slug = ?', [$slug])) {
            Response::error('Bu slug zaten kullanılıyor', 'SLUG_EXISTS', 409);
        }

        $bookId = Database::insert(
            'INSERT INTO books (slug, category_id, status, is_free, preview_pages, default_lang, total_pages, sort_order)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $slug,
                Request::input('categoryId') ? (int)Request::input('categoryId') : null,
                Request::input('status', 'draft'),
                Request::input('isFree', 0) ? 1 : 0,
                (int)Request::input('previewPages', 10),
                $lang,
                (int)Request::input('totalPages', 0),
                (int)Request::input('sortOrder', 0),
            ]
        );

        // İlk çeviriyi ekle
        Database::insert(
            'INSERT INTO book_translations (book_id, lang, title, description, author, publisher, tags)
             VALUES (?, ?, ?, ?, ?, ?, ?)',
            [
                $bookId, $lang, $title,
                Request::input('description', ''),
                Request::input('author', ''),
                Request::input('publisher', ''),
                Request::input('tags', ''),
            ]
        );

        Response::success(['id' => (int)$bookId, 'slug' => $slug], 'Kitap oluşturuldu', 201);
    }

    /**
     * GET /api/admin/books/{id}
     */
    public function show(array $params): void
    {
        $id = (int)$params['id'];

        $book = Database::fetch('SELECT * FROM books WHERE id = ?', [$id]);
        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        $translations = Database::fetchAll(
            'SELECT * FROM book_translations WHERE book_id = ?', [$id]
        );

        $assets = Database::fetchAll(
            'SELECT * FROM book_assets WHERE book_id = ? ORDER BY lang, format, version DESC', [$id]
        );

        $chunkIndex = Database::fetchAll(
            'SELECT * FROM book_chunk_index WHERE book_id = ? ORDER BY lang, version, chunk_no', [$id]
        );

        $book['translations'] = $translations;
        $book['assets'] = $assets;
        $book['chunkIndex'] = $chunkIndex;

        Response::success($book);
    }

    /**
     * PUT /api/admin/books/{id}
     */
    public function update(array $params): void
    {
        $id = (int)$params['id'];
        $data = Request::body();

        $book = Database::fetch('SELECT id FROM books WHERE id = ?', [$id]);
        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        $fields = [];
        $values = [];
        $allowed = ['slug', 'category_id', 'status', 'is_free', 'preview_pages', 'default_lang', 'total_pages', 'sort_order'];
        $map = [
            'slug' => 'slug', 'categoryId' => 'category_id', 'status' => 'status',
            'isFree' => 'is_free', 'previewPages' => 'preview_pages',
            'defaultLang' => 'default_lang', 'totalPages' => 'total_pages', 'sortOrder' => 'sort_order',
        ];

        foreach ($map as $input => $column) {
            if (isset($data[$input])) {
                $fields[] = "{$column} = ?";
                $values[] = $data[$input];
            }
        }

        if (!empty($fields)) {
            $values[] = $id;
            Database::execute(
                'UPDATE books SET ' . implode(', ', $fields) . ', updated_at = NOW() WHERE id = ?',
                $values
            );
        }

        Response::success(null, 'Kitap güncellendi');
    }

    /**
     * DELETE /api/admin/books/{id}
     */
    public function destroy(array $params): void
    {
        $id = (int)$params['id'];

        $book = Database::fetch('SELECT id FROM books WHERE id = ?', [$id]);
        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        Database::execute('DELETE FROM books WHERE id = ?', [$id]);

        Response::success(null, 'Kitap silindi');
    }

    /**
     * GET /api/admin/books/{id}/translations
     */
    public function translations(array $params): void
    {
        $id = (int)$params['id'];
        $translations = Database::fetchAll(
            'SELECT * FROM book_translations WHERE book_id = ?', [$id]
        );
        Response::success($translations);
    }

    /**
     * POST /api/admin/books/{id}/translations
     */
    public function addTranslation(array $params): void
    {
        $bookId = (int)$params['id'];
        $data = Request::validate(['lang', 'title']);

        // Kitap var mı?
        if (!Database::fetch('SELECT id FROM books WHERE id = ?', [$bookId])) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Dil zaten ekli mi?
        if (Database::fetch('SELECT id FROM book_translations WHERE book_id = ? AND lang = ?', [$bookId, $data['lang']])) {
            Response::error('Bu dil zaten eklenmiş', 'TRANSLATION_EXISTS', 409);
        }

        $id = Database::insert(
            'INSERT INTO book_translations (book_id, lang, title, description, author, publisher, tags, seo_title, seo_desc)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $bookId, $data['lang'], $data['title'],
                Request::input('description', ''),
                Request::input('author', ''),
                Request::input('publisher', ''),
                Request::input('tags', ''),
                Request::input('seoTitle', ''),
                Request::input('seoDesc', ''),
            ]
        );

        Response::success(['id' => (int)$id], 'Çeviri eklendi', 201);
    }

    /**
     * PUT /api/admin/translations/{id}
     */
    public function updateTranslation(array $params): void
    {
        $id = (int)$params['id'];
        $data = Request::body();

        $translation = Database::fetch('SELECT id FROM book_translations WHERE id = ?', [$id]);
        if (!$translation) {
            Response::error('Çeviri bulunamadı', 'TRANSLATION_NOT_FOUND', 404);
        }

        $fields = [];
        $values = [];
        $map = ['title', 'description', 'author', 'publisher', 'tags', 'seoTitle' => 'seo_title', 'seoDesc' => 'seo_desc'];

        foreach ($map as $key => $column) {
            $inputKey = is_int($key) ? $column : $key;
            $col = is_int($key) ? $column : $column;
            if (isset($data[$inputKey])) {
                $fields[] = "{$col} = ?";
                $values[] = $data[$inputKey];
            }
        }

        if (!empty($fields)) {
            $values[] = $id;
            Database::execute(
                'UPDATE book_translations SET ' . implode(', ', $fields) . ', updated_at = NOW() WHERE id = ?',
                $values
            );
        }

        Response::success(null, 'Çeviri güncellendi');
    }
}
