// E-Kitap Admin Panel - JavaScript
const API_BASE = window.location.origin + '/api';

// State
const state = {
    user: null,
    token: null,
    currentPage: 'dashboard',
    theme: localStorage.getItem('theme') || 'dark'
};

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    initTheme();
    checkAuth();
    setupEventListeners();
});

// Theme
function initTheme() {
    document.documentElement.setAttribute('data-theme', state.theme);
}

function toggleTheme() {
    state.theme = state.theme === 'dark' ? 'light' : 'dark';
    localStorage.setItem('theme', state.theme);
    document.documentElement.setAttribute('data-theme', state.theme);
}

// Auth Check
function checkAuth() {
    const token = localStorage.getItem('accessToken');
    if (token) {
        state.token = token;
        fetchUserData();
    } else {
        showLoginScreen();
    }
}

function showLoginScreen() {
    document.getElementById('login-screen').style.display = 'flex';
    document.getElementById('app').style.display = 'none';
}

function showApp() {
    document.getElementById('login-screen').style.display = 'none';
    document.getElementById('app').style.display = 'flex';
}

// Event Listeners
function setupEventListeners() {
    // Login
    document.getElementById('login-form').addEventListener('submit', handleLogin);

    // Logout
    document.getElementById('logout-btn').addEventListener('click', handleLogout);

    // Navigation
    document.querySelectorAll('.nav-item').forEach(item => {
        item.addEventListener('click', (e) => {
            e.preventDefault();
            const page = item.dataset.page;
            navigateTo(page);
        });
    });

    // Sidebar toggle
    document.getElementById('sidebar-toggle').addEventListener('click', () => {
        document.getElementById('sidebar').classList.toggle('collapsed');
    });

    // Mobile menu
    document.getElementById('mobile-menu-btn').addEventListener('click', () => {
        document.getElementById('sidebar').classList.toggle('mobile-open');
    });

    // Theme toggle
    document.getElementById('theme-toggle').addEventListener('click', toggleTheme);

    // Modal close
    document.getElementById('modal-close').addEventListener('click', closeModal);
    document.getElementById('modal-overlay').addEventListener('click', (e) => {
        if (e.target.id === 'modal-overlay') closeModal();
    });
}

// API Helper
async function apiCall(endpoint, options = {}) {
    const headers = {
        'Content-Type': 'application/json',
        ...options.headers
    };

    if (state.token) {
        headers['Authorization'] = `Bearer ${state.token}`;
    }

    try {
        const response = await fetch(API_BASE + endpoint, {
            ...options,
            headers
        });

        const data = await response.json();

        if (!response.ok) {
            throw new Error(data.error?.message || 'Bir hata oluştu');
        }

        return data;
    } catch (error) {
        showToast(error.message, 'error');
        throw error;
    }
}

// Login
async function handleLogin(e) {
    e.preventDefault();

    const email = document.getElementById('login-email').value;
    const password = document.getElementById('login-password').value;
    const btn = document.getElementById('login-btn');
    const errorDiv = document.getElementById('login-error');

    btn.disabled = true;
    btn.textContent = 'Giriş yapılıyor...';
    errorDiv.style.display = 'none';

    try {
        const data = await apiCall('/auth/login', {
            method: 'POST',
            body: JSON.stringify({ email, password })
        });

        state.token = data.data.accessToken;
        state.user = data.data.user;

        localStorage.setItem('accessToken', data.data.accessToken);
        localStorage.setItem('refreshToken', data.data.refreshToken);

        showApp();
        loadDashboard();
        updateUserInfo();
        showToast('Giriş başarılı!', 'success');

    } catch (error) {
        errorDiv.textContent = error.message;
        errorDiv.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<span class="material-icons-round">login</span> Giriş Yap';
    }
}

// Logout
async function handleLogout() {
    try {
        await apiCall('/auth/logout', { method: 'POST' });
    } catch (e) { }

    localStorage.removeItem('accessToken');
    localStorage.removeItem('refreshToken');
    state.token = null;
    state.user = null;
    showLoginScreen();
    showToast('Çıkış yapıldı', 'success');
}

// Fetch User Data
async function fetchUserData() {
    try {
        const data = await apiCall('/me');
        state.user = data.data.user;
        updateUserInfo();
        showApp();
        loadDashboard();
    } catch (error) {
        showLoginScreen();
    }
}

function updateUserInfo() {
    if (state.user) {
        document.getElementById('user-name').textContent = state.user.name;
        document.getElementById('user-role').textContent = state.user.roles[0] || 'user';
        document.getElementById('user-avatar').textContent = state.user.name.charAt(0).toUpperCase();
    }
}

// Navigation
function navigateTo(page) {
    state.currentPage = page;

    // Update nav
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.toggle('active', item.dataset.page === page);
    });

    // Update pages
    document.querySelectorAll('.page').forEach(p => {
        p.classList.toggle('active', p.id === 'page-' + page);
    });

    // Update title
    const titles = {
        dashboard: 'Dashboard',
        books: 'Kitaplar',
        users: 'Kullanıcılar',
        subscriptions: 'Abonelikler',
        categories: 'Kategoriler',
        reports: 'Raporlar',
        settings: 'Ayarlar'
    };
    document.getElementById('page-title').textContent = titles[page] || page;

    // Load page data
    switch (page) {
        case 'dashboard':
            loadDashboard();
            break;
        case 'books':
            loadBooks();
            break;
        case 'users':
            loadUsers();
            break;
        case 'subscriptions':
            loadSubscriptionReport();
            break;
        case 'reports':
            loadPopularBooksReport();
            break;
    }
}

// Load Dashboard
async function loadDashboard() {
    try {
        const data = await apiCall('/admin/dashboard');
        const stats = data.data.stats;

        document.getElementById('stat-users').textContent = stats.totalUsers;
        document.getElementById('stat-books').textContent = stats.publishedBooks;
        document.getElementById('stat-subs').textContent = stats.activeSubscriptions;
        document.getElementById('stat-readers').textContent = stats.activeReaders30d;

        // Popular books
        const tbody = document.querySelector('#popular-books-table tbody');
        tbody.innerHTML = data.data.popularBooks.map(book => `
            <tr>
                <td>${book.title || 'Başlıksız'}</td>
                <td>${book.view_count}</td>
            </tr>
        `).join('');

        // Recent users
        const userTbody = document.querySelector('#recent-users-table tbody');
        userTbody.innerHTML = data.data.recentUsers.map(user => `
            <tr>
                <td>${user.name} (${user.email})</td>
                <td>${new Date(user.created_at).toLocaleDateString('tr-TR')}</td>
            </tr>
        `).join('');

    } catch (error) {
        console.error('Dashboard error:', error);
    }
}

// Load Books
async function loadBooks(page = 1) {
    try {
        const status = document.getElementById('filter-book-status')?.value || '';
        const params = new URLSearchParams({ page, limit: 20 });
        if (status) params.append('status', status);

        const data = await apiCall('/admin/books?' + params.toString());

        const tbody = document.querySelector('#books-table tbody');
        tbody.innerHTML = data.data.map(book => `
            <tr>
                <td>${book.id}</td>
                <td>
                    ${book.cover_path
                ? `<img src="/storage/${book.cover_path}" width="40" height="60" style="border-radius:4px;">`
                : '<div style="width:40px;height:60px;background:#333;border-radius:4px;"></div>'}
                </td>
                <td>${book.title || 'Başlıksız'}</td>
                <td>${book.author || '-'}</td>
                <td>
                    <span class="badge badge-${book.status === 'published' ? 'success' : 'secondary'}">
                        ${book.status}
                    </span>
                </td>
                <td>${book.formats?.join(', ') || '-'}</td>
                <td>${book.translationCount || 0}</td>
                <td>
                    <button class="btn-icon" onclick="editBook(${book.id})" title="Düzenle">
                        <span class="material-icons-round">edit</span>
                    </button>
                    <button class="btn-icon" onclick="deleteBook(${book.id})" title="Sil">
                        <span class="material-icons-round">delete</span>
                    </button>
                </td>
            </tr>
        `).join('');

        // Pagination
        renderPagination('books-pagination', data.meta, (p) => loadBooks(p));

    } catch (error) {
        console.error('Books error:', error);
    }
}

// Load Users
async function loadUsers(page = 1) {
    try {
        const q = document.getElementById('filter-user-search')?.value || '';
        const params = new URLSearchParams({ page, limit: 20 });
        if (q) params.append('q', q);

        const data = await apiCall('/admin/users?' + params.toString());

        const tbody = document.querySelector('#users-table tbody');
        tbody.innerHTML = data.data.map(user => `
            <tr>
                <td>${user.id}</td>
                <td>${user.name}</td>
                <td>${user.email}</td>
                <td>${user.roles?.join(', ') || '-'}</td>
                <td>
                    ${user.subscription?.is_active
                ? '<span class="badge badge-success">Aktif</span>'
                : '<span class="badge badge-secondary">-</span>'}
                </td>
                <td>${user.last_login_at ? new Date(user.last_login_at).toLocaleDateString('tr-TR') : '-'}</td>
                <td>
                    <button class="btn-icon" onclick="editUser(${user.id})" title="Düzenle">
                        <span class="material-icons-round">edit</span>
                    </button>
                </td>
            </tr>
        `).join('');

        renderPagination('users-pagination', data.meta, (p) => loadUsers(p));

    } catch (error) {
        console.error('Users error:', error);
    }
}

// Load Subscription Report
async function loadSubscriptionReport() {
    try {
        const data = await apiCall('/admin/reports/subscriptions');
        const report = data.data;

        const html = `
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-info">
                        <span class="stat-value">${report.expiringCount}</span>
                        <span class="stat-label">7 Gün İçinde Sona Erecek</span>
                    </div>
                </div>
            </div>

            <h4 style="margin-top:24px;">Ürüne Göre Dağılım</h4>
            <table class="table">
                <thead>
                    <tr>
                        <th>Ürün ID</th>
                        <th>Toplam</th>
                        <th>Aktif</th>
                    </tr>
                </thead>
                <tbody>
                    ${report.byProduct.map(p => `
                        <tr>
                            <td>${p.product_id}</td>
                            <td>${p.total}</td>
                            <td>${p.active}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;

        document.getElementById('subscription-report').innerHTML = html;

    } catch (error) {
        console.error('Subscription report error:', error);
    }
}

// Load Popular Books Report
async function loadPopularBooksReport() {
    try {
        const data = await apiCall('/admin/reports/popular-books');

        const html = `
            <table class="table">
                <thead>
                    <tr>
                        <th>Kitap</th>
                        <th>Yazar</th>
                        <th>Görüntülenme</th>
                        <th>Okuyucu</th>
                        <th>Ort. İlerleme</th>
                    </tr>
                </thead>
                <tbody>
                    ${data.data.map(book => `
                        <tr>
                            <td>${book.title || 'Başlıksız'}</td>
                            <td>${book.author || '-'}</td>
                            <td>${book.view_count}</td>
                            <td>${book.readers || 0}</td>
                            <td>${book.avg_progress ? Math.round(book.avg_progress) + '%' : '-'}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;

        document.getElementById('reports-content').innerHTML = html;

    } catch (error) {
        console.error('Reports error:', error);
    }
}

// Pagination
function renderPagination(containerId, meta, onPageChange) {
    const container = document.getElementById(containerId);
    if (!container || !meta) return;

    const { page, totalPages } = meta;
    let html = '';

    // Previous
    html += `<button ${page <= 1 ? 'disabled' : ''} onclick="(${onPageChange})(${page - 1})">Önceki</button>`;

    // Pages
    for (let i = Math.max(1, page - 2); i <= Math.min(totalPages, page + 2); i++) {
        html += `<button class="${i === page ? 'active' : ''}" onclick="(${onPageChange})(${i})">${i}</button>`;
    }

    // Next
    html += `<button ${page >= totalPages ? 'disabled' : ''} onclick="(${onPageChange})(${page + 1})">Sonraki</button>`;

    container.innerHTML = html;
}

// Modal
function showModal(title, bodyHtml, footerHtml = '') {
    document.getElementById('modal-title').textContent = title;
    document.getElementById('modal-body').innerHTML = bodyHtml;
    document.getElementById('modal-footer').innerHTML = footerHtml;
    document.getElementById('modal-overlay').classList.add('active');
}

function closeModal() {
    document.getElementById('modal-overlay').classList.remove('active');
}

// Toast
function showToast(message, type = 'info') {
    const container = document.getElementById('toast-container');
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    const icon = {
        success: 'check_circle',
        error: 'error',
        warning: 'warning',
        info: 'info'
    }[type] || 'info';

    toast.innerHTML = `
        <span class="material-icons-round">${icon}</span>
        <span>${message}</span>
    `;

    container.appendChild(toast);

    setTimeout(() => {
        toast.style.animation = 'fadeOut 0.3s ease-out';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// CRUD Operations (placeholders)
function editBook(id) {
    showModal('Kitap Düzenle', `<p>Kitap ID: ${id}</p><p>Düzenleme formu yakında...</p>`);
}

function deleteBook(id) {
    if (confirm('Bu kitabı silmek istediğinizden emin misiniz?')) {
        showToast('Silme özelliği yakında...', 'info');
    }
}

function editUser(id) {
    showModal('Kullanıcı Düzenle', `<p>Kullanıcı ID: ${id}</p><p>Düzenleme formu yakında...</p>`);
}
